﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.ServiceModel;

namespace VA.PPMS.CRM.Plugins
{
    public class ProviderUpdate : IPlugin
    {
        private const string PluginName = "ProviderUpdate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            ITracingService tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity provider = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (provider.LogicalName != "account")
                    return;

                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Get related entities to deactivate
                    tracingService.Trace("Retrieve Provider and related entities");
                    Entity entity = GetProvider(service, provider.Id, tracingService);
                    if (entity == null)
                    {
                        tracingService.Trace("Failed to retrieve provider related entities");
                        return;
                    }
                    tracingService.Trace("Provider Retrieved");

                    List<SetStateRequest> requests = new List<SetStateRequest>();

                    // handle event based on message type
                    tracingService.Trace("Check status");
                    var isActive = PpmsHelper.IsActive(entity);
                    int state = 0;
                    int statuscode = 0;

                    if (isActive)
                    {
                        tracingService.Trace("Provider in Active Status");
                        state = (int)PpmsHelper.AccountState.Active;
                        statuscode = (int)PpmsHelper.Account_StatusCode.Active;
                    }
                    else
                    {
                        tracingService.Trace("Provider in Inactive Status");
                        state = (int)PpmsHelper.AccountState.Inactive;
                        statuscode = (int)PpmsHelper.Account_StatusCode.Inactive;
                    }

                    tracingService.Trace("Adding Set State Requests to the related entities");

                    // other identifier
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_ppms_otherprovideridentifier", state, statuscode));

                    // npi
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_ppms_provideridentifier_Provider", state, statuscode));

                    // provider service
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_ppms_providerservice", state, statuscode));

                    // board certification
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_ppms_boardcertification", state, statuscode));

                    // medical education
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_ppms_medicaleducation", state, statuscode));

                    // DEA Schedule
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_deascheduleprivilege", state, statuscode));

                    // license
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_providerlicensure", state, statuscode));

                    // authorized official
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_organizationauthorizedofficial", state, statuscode));

                    // provider other name
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_ppms_othername", state, statuscode));

                    // specialty
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_account_ppms_providertaxonomy", state, statuscode));

                    // agreements - only if changing to inactive
                    if (state == (int)PpmsHelper.AccountState.Inactive)
                        requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_ppm_provideragreement", state, GetAgreementStatus(state)));

                    if (requests != null)
                    {
                        tracingService.Trace("Exectuting Set State Requests");
                        foreach (var request in requests)
                        {
                            service.Execute(request);
                        }
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }

        private Entity GetProvider(IOrganizationService service, Guid providerId, ITracingService tracingService)
        {
            var request = new RetrieveRequest();

            request.Target = new EntityReference("account", providerId);
            request.ColumnSet = new ColumnSet(new string[] { "accountid", "name", "statecode" });
            request.RelatedEntitiesQuery = new RelationshipQueryCollection();

            //Retrieve related entities
            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_ppms_otherprovideridentifier"),
                new QueryExpression("ppms_otherprovideridentifier")
                {
                    ColumnSet = new ColumnSet("ppms_otherprovideridentifierid", "ppms_name")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_ppms_provideridentifier_Provider"),
                new QueryExpression("ppms_provideridentifier")
                {
                    ColumnSet = new ColumnSet("ppms_provideridentifierid", "ppms_provideridentifier")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_ppms_providerservice"),
                new QueryExpression("ppms_providerservice")
                {
                    ColumnSet = new ColumnSet("ppms_providerserviceid", "ppms_name")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_ppms_boardcertification"),
                new QueryExpression("ppms_boardcertification")
                {
                    ColumnSet = new ColumnSet("ppms_boardcertificationid", "ppms_name")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_ppms_medicaleducation"),
                new QueryExpression("ppms_medicaleducation")
                {
                    ColumnSet = new ColumnSet("ppms_medicaleducationid", "ppms_name")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_deascheduleprivilege"),
                new QueryExpression("ppms_deascheduleprivilege")
                {
                    ColumnSet = new ColumnSet("ppms_deascheduleprivilegeid", "ppms_deanumber")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_providerlicensure"),
                new QueryExpression("ppms_providerlicensure")
                {
                    ColumnSet = new ColumnSet("ppms_providerlicensureid", "ppms_name")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_organizationauthorizedofficial"),
                new QueryExpression("ppms_organizationauthorizedofficial")
                {
                    ColumnSet = new ColumnSet("ppms_organizationauthorizedofficialid", "ppms_name")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_ppms_othername"),
                new QueryExpression("ppms_othername")
                {
                    ColumnSet = new ColumnSet("ppms_othernameid", "ppms_name")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_ppms_providertaxonomy"),
                new QueryExpression("ppms_providertaxonomy")
                {
                    ColumnSet = new ColumnSet("ppms_providertaxonomyid", "ppms_codedspecialty")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_ppm_provideragreement"),
                new QueryExpression("ppms_provideragreement")
                {
                    ColumnSet = new ColumnSet("ppms_provideragreementid", "ppms_agreementid")
                }
            );

            //Get response
            tracingService.Trace("Excecuting request to retrieve Provider and relationships");
            var response = (RetrieveResponse)service.Execute(request);
            if (response != null)
                return response.Entity;

            return null;
        }

        /// <summary>
        /// Determine new status code for agreement based on the new state
        /// </summary>
        /// <param name="agreementState">State of the agreement</param>
        /// <returns>If state = Active, Active, else Inactive - Ineligible</returns>
        private int GetAgreementStatus(int agreementState)
        {
            return agreementState == (int)PpmsHelper.AccountState.Active
                ? (int)VA.PPMS.Context.ppms_provideragreement_StatusCode.Active
                : (int)VA.PPMS.Context.ppms_provideragreement_StatusCode.Ineligible_Inactive;
        }
    }
}
